/*  Defines the object "mymetro" which is similar to the standard 
	"metro" Max object. The metro object has 2 inlets and 2 outlets. 
	
	"bang" 	in left inlet starts metronome
	"stop" 	in left inlet stops metronome
	integer	in right inlet sets tempo in ms
	
	left output sends bangs at each metronome interval
	right outlet outputs current time
	
	The object also posts messages to the Max window indicating the current state of
	mymetro.
*/

#include "ext.h"		// Required for all Max external objects

#define DEFAULT_TEMPO 	1000
#define MIN_TEMPO 	     40

typedef struct _metro			/* data structure for this object */
{
	t_object 	m_ob;			/* must always be the first field; used by Max */
	void 	*m_clock;		/* pointer to clock object */
	long 	m_interval;		/* tempo in milliseconds */
	void 	*m_bang_outlet;	/* pointers to bang outlet */
	void 	*m_time_outlet;	/* pointers to time outlet */
} t_metro;

void *metro_new(long value);			
void metro_in1(t_metro *m, long value);	
void metro_bang(t_metro *m);				
void metro_assist(t_metro *m, t_object *b, long msg, long arg, char *s);
void metro_free(t_metro *m);
void metro_stop(t_metro *m);
void clock_function(t_metro *m);

void *class;		// Required. Global pointing to this class 

int main(void)
{
	
	/* set up our class: create a class definition */
	setup((t_messlist **)&class, (method)metro_new, (method)metro_free, 
			(short)sizeof(t_metro), 0L, A_DEFLONG, 0);
		
	/* bind method "metro_bang" to the "bang" message */
	addbang((method)metro_bang);
		
	/* bind method "metro_in1" to int's received in the right inlet */	
	addinx((method)metro_in1,1);
	
	/* bind method "metro_stop" to the "stop" message" */
	addmess((method)metro_stop,"stop",0);

	/* bind method "metro_assist" to the assistance message" */
	addmess((method)metro_assist,"assist",A_CANT,0);

	/* add class to the New object list */
	finder_addclass("All Objects","mymetro");
	return(0);
}

/**********************************************************************************
metro_new(long value)

inputs:		value - the integer from the typed in argument in the object box
description:	creates a new instance of this class metro.
returns:		pointer to new instance
*************************************************************************************/
void*metro_new(long value)
{
	t_metro *m;

	m = (t_metro *)newobject(class); // create the new instance and return a pointer to it

	if (value > MIN_TEMPO)			// initialize the subtraction value
	{
		m->m_interval = value;		// save tempo arguement from box
		post("mymetro tempo set to %ld", value);
	}
	else
	{
		m->m_interval = DEFAULT_TEMPO;	// set to default tempo
		post("mymetro set to default tempo of %ld ms", DEFAULT_TEMPO);
	}
	m->m_clock = clock_new(m, (method)clock_function);  // create the metronome clock

	intin(m, 1);					// create the right inlet
	m->m_time_outlet = intout(m);	// create right outlet for time
	m->m_bang_outlet = bangout(m);	// create left outlet for ticks
	
	return(m);
}
/*************************************************************************************
metro_in1(t_metro *m, long value)

inputs:	m	  -- pointer to our object
		value   -- value received in the inlet
description:	  stores the new metronome tempo value
*************************************************************************************/
void metro_in1(t_metro *m, long value)
{
	m->m_interval = value;		// store the new metronome interval
	post("metronome tempo changed to %ld", value);
}
 /*************************************************************************************
void *metro_bang(t_metro *m)

inputs:		m -- pointer to our object	
description:	method called when bang is received: it starts the metronome
*************************************************************************************/
void metro_bang(t_metro *m)
{
	long time;
	
	time = gettime();				// get current time
	clock_delay(m->m_clock, 0L); 	// set clock to go off now 
	post("clock started at %ld", time);
}
/*************************************************************************************
void *metro_stop(t_metro *m)

inputs:		m -- pointer to our object	
description:	method called when myMetro receives "stop" message. Stops the metronome
*************************************************************************************/
void metro_stop(t_metro *m)
{
	long time;
	
	time = gettime();			// get current time
	clock_unset(m->m_clock);	// remove the clock routine from the scheduler
	outlet_int(m->m_time_outlet, time);
	post("metronome stopped at %ld", time);
}


/*************************************************************************************
void clock_function(t_metro *m)

inputs:		m -- pointer to our object
description:	method called when clock goes off: it outputs a bang to be sent to the 			
outlet and resets the clock  to go off after the next interval.
*************************************************************************************/
void clock_function(t_metro *m)
{
	long time;

	time = gettime();		// get current time
	clock_delay(m->m_clock, m->m_interval);	// schedule another metronome click
	outlet_bang(m->m_bang_outlet);			// send out a bang
	outlet_int(m->m_time_outlet, time);		// send current time to right outlet
	post("clock_function %ld", time);   // demonstrates the interrupt call (in Overdrive)
	
}


/*************************************************************************************
metro_free(t_metro *m)

inputs:		m -- pointer to our object
description:	method called when t_metro objects is destroyed. It is used to free memory 			
allocated to the clock.
*************************************************************************************/
void metro_free(t_metro *m)
{
	clock_unset(m->m_clock);	// remove the clock routine from the scheduler
	clock_free(m->m_clock); 	// free the clock memory
}

void metro_assist(t_metro *m, t_object *b, long msg, long arg, char *s)
{	
	// copy the appropriate message to the destination string
	//assist_string(ResourceID, msg, arg, 1L, 3L, s);
}
